
% This script reads imaris files and measures the total original protein
% and RNA intensity. Written by Ingrid Brust-Mascher and Xueer Jiang, Jao
% lab, University of California, Davis (April 2019). Publications that have
% utilized this script should acknowledge the resource and cite our
% Bio-Protocol Paper.





clear
% define path: location of a folder containing the processed imaris images
fpath='/Users/jao_lab/Documents/MATLAB/bio_protocol_images_control';       

% define protein surface of interest and RNA spot of interest for
% quantification
protein_surface_number = 2;
spot_number = 1;

% define the channel number of original protein surface and original RNA
% spot of interest
ori_protein_channel_number = '5';
ori_RNA_channel_number = '4';

% define intensity sum channel number for generalization
intensity_sum = 'Intensity Sum Channel';
ori_intensity_sum_channel_protein = [intensity_sum, ' ', ori_protein_channel_number];
ori_intensity_sum_channel_RNA = [intensity_sum, ' ', ori_RNA_channel_number];

% use dir function to list folder contents and select imaris files
ImarisImages=dir(fullfile(fpath,'*.ims'));

% define numbers of files
NumberImages=length(ImarisImages); 

% define current total protein and RNA sum intensity as empty matrix and
% result as matrix of zeros
TotalProIntensity = [];

TotalRNAIntensity = [];

result = zeros(NumberImages,7);

% Create a forloop to define filenames for analysis
for i = 1:NumberImages 
      
    %filename to analyze
    filename=ImarisImages(i).name;                        
    
    %create filenames
    xPath=fullfile(fpath,filename);          
    
    %read imaris files using ImarisReader function
    imsObj=ImarisReader(xPath);             
   
    % index surface statisitics of protein of interest
    SurfaceStatistics=imsObj.Surfaces(protein_surface_number).GetStatistics; 
    
    % index spot statisitics of RNA of interest
    SpotStatistics=imsObj.Spots(spot_number).GetStatistics;       
    
    % define number of surface statisitics
    NumberSurStats=length(SurfaceStatistics);           
    
    % define number of spot statisitics
    NumberSpotStats=length(SpotStatistics);              
    
    for k = 1:NumberSurStats
        
        %select the channel with original surface intensity
         if strcmp(SurfaceStatistics(k).Name,ori_intensity_sum_channel_protein) 
             
            % extract the values of original surface intensity for each
            % surface
            SurfaceOriginalIntensity=SurfaceStatistics(k).Value;   
            
         end % conditional branching to get original surface intensity in each image       
         
    end  % for number of protein surfaces per image
    
    
    for j = 1:NumberSpotStats
        
        %select the channel with original RNA spot intensity
        if strcmp(SpotStatistics(j).Name,ori_intensity_sum_channel_RNA) 
            
           % extract the values of original RNA spot intensity for each RNA
           % spot
           RNAOriginalIntensity=SpotStatistics(j).Value; 
           
        end % conditional branching to get original RNA spot intensity in each image  
        
    end % for number of RNA spots per image
   
   
    % sum all protein surface original intensities per image
    Sum_Surface_OriIntensity = sum(SurfaceOriginalIntensity); 
    
    % sum all RNA spot original intensities per image
    Sum_RNASpot_OriIntensity =  sum(RNAOriginalIntensity); 
    
    % save image number or order into result
    result(i,1) = i;   
    
    % save individual original protein intensity of each image into result
    result(i,2) = Sum_Surface_OriIntensity;  
    
    % save individual original RNA intensity of each image into result
    result(i,5) = Sum_RNASpot_OriIntensity;      
   
    % collect surface original intensities as an vector
    TotalProIntensity = [TotalProIntensity Sum_Surface_OriIntensity]; 
    
    % collect RNA spot original intensities as an vector
    TotalRNAIntensity = [TotalRNAIntensity Sum_RNASpot_OriIntensity]; 
    
end % for number of images

% save mean of original protein intensity of all images
result(1,3) = mean(TotalProIntensity);      

% save standard deviation of original protein intensity of all images
result(1,4) = std(TotalProIntensity);     

% save mean of original RNA intensity of all images
result(1,6) = mean(TotalRNAIntensity); 

% save standard deviation of original RNA intensity of all images
result(1,7) = std(TotalRNAIntensity);         
    
    
% convert matrix to table and define column names
 table_result = array2table(result);
 
% define variable names for the table
 table_result.Properties.VariableNames{'result1'} = 'image_order';
 table_result.Properties.VariableNames{'result2'} = 'Ori_Protein_Intensity';
 table_result.Properties.VariableNames{'result3'} = 'Mean_Protein_Intensity';
 table_result.Properties.VariableNames{'result4'} = 'Std_Protein_Intensity';
 table_result.Properties.VariableNames{'result5'} = 'Ori_RNA_Intensity';
 table_result.Properties.VariableNames{'result6'} = 'Mean_RNA_Intensity';
 table_result.Properties.VariableNames{'result7'} = 'Std_RNA_Intensity';
    
% define output name and format
outputfile = 'original total protein and total RNA intensity.csv'; 

% define output path
OutputPath = fullfile(fpath,outputfile);

% output as csv table spreadsheet
writetable(table_result,OutputPath); 
